// --------------------------------- lpssocket.h -------------------------------------------
//
//  Include file for Linux LabBrick LPS socket definitions
//
//  (c) 2020-2025 by Vaunix Technology Corporation, all rights reserved
//
//  NJB  Version 1.0 LPS Ethernet Socket Device Driver
//
//-----------------------------------------------------------------------------
#ifndef __LPSSOCKET_H__
#define __LPSSOCKET_H__

/// ---------- Include headers ----------------
#include "lpsdrvr.h"

//*****************************************************************************
//
// If building with a C++ compiler, make all of the definitions in this header
// have a C binding.
//
//*****************************************************************************
#ifdef __cplusplus
extern "C"
{
#endif

// Hardware type values by device name (used in lps[DeviceId].DevType)
#define LPS_802_8	  6
#define LPS_402_8 	8
#define LPS_202_8 	9


#define STATUS_PROFILE_ACTIVE 0x80		// MASK: A profile is playing

// Feature bits for the feature DWORD 
#define DEFAULT_FEATURES	0x00000000
#define HAS_BIDIR_RAMPS		0x00000001
#define HAS_PROFILES		0x00000002
#define HAS_HIRES			0x00000004
#define HAS_4CHANNELS		0x00000008
#define HAS_8CHANNELS		0x00000010
#define HAS_LONG_PROFILE	0x00000020
#define HAS_MCHANNELS		0x00000040

// Bit masks and equates for the Sweep command byte (stored in Sweep_mode, and reported also in Status)
#define SWP_DIRECTION		0x04		// MASK: bit = 0 for sweep or ramp up, 1 for sweep or ramp down
#define SWP_CONTINUOUS		0x02		// MASK: bit = 1 for continuous sweeping
#define SWP_ONCE			0x01		// MASK: bit = 1 for single sweep
#define SWP_BIDIR			0x10		// MASK: bit = 0 for ramp style sweep, 1 for triangle style sweep

// Mode Bit Masks
#define MODE_SWEEP 		0x0000001F 	// bottom 5 bits are used to keep the sweep control bits

// HID report equates
#define HR_BLOCKSIZE 6				// size of the block of bytes buffer in our HID report
#define HID_REPORT_LENGTH 8			// use an 8 byte report..

// Misc. constants used for the Vaunix devices
#define VNX_MIN_DWELLTIME 1


// Misc commands to send to the device
#define VNX_SET				0x80
#define VNX_GET				0x00	// the set and get bits are or'd into the msb of the command byte

// ---------------------- Phase shifter commands ------------------------
#define VNX_PHASEANGLE		0x50	// phase shift angle in degrees (DWORD)
#define VNX_FREQUENCY		0x04	// working frequency in 100Khz units (set in 1MHz increments)
#define VNX_MINFREQUENCY	0x20	// minimum working frequency in 100Khz units (set in 1MHz increments)
#define VNX_MAXFREQUENCY	0x21	// maximum working frequency in 100Khz units (set in 1MHz increments)

// ----------------- Phase shifter ramp commands ------------------------
#define VNX_SWEEP			0x09	// command to start/stop sweep, data = 01 for single sweep, 00 to stop
									// sweeping, and 02 for continuous sweeping.

#define VNX_ASTART			0x30	// initial value for phase ramp
#define VNX_ASTOP			0x31	// final value for phase ramp

#define VNX_ASTEP			0x32	// step size for first part of phase ramp
#define VNX_ASTEP2			0x38	// step size for second part of phase ramp

#define VNX_ADWELL			0x33	// dwell time on each step for first part of phase ramp
#define VNX_ADWELL2			0x37	// dwell time on each step for second part of phase ramp

#define VNX_AIDLE			0x36	// idle time between phase ramps in milliseconds
#define VNX_AHOLD			0x39	// hold time between parts of the ramps in milliseconds

#define VNX_SETPROFILE		0x3A	// set/get profile values, first byte is unused
									// second data byte is the index (0 based)
									// the third is the angle value for that profile entry
									
#define VNX_PROFILECOUNT	0x3B	// number of elements in the profile, 1 to PROFILE_MAX = 100

#define VNX_PROFILEDWELL	0x3C	// dwell time for each profile element

#define VNX_PROFILEIDLE		0x3D	// idle time at the end of each repeating profile 

#define VNX_PROFILEINDEX	0x3F	// the current value of the profile index, mostly used to read the index...

#define VNX_SAVEPAR			0x0C	// command to save user parameters to flash, data bytes must be
									// set to 0x42, 0x55, 0x31 as a key to enable the flash update
									// all of the above settings are saved (angle, sweep parameters, etc.)

#define VNX_MINPHASE		0x34	// used to get the minimum phase shift angle which is 0 for every case now

#define VNX_MAXPHASE		0x35	// get the maximum phase shift

#define VNX_GETSERNUM		0x1F	// get the serial number, value is a DWORD

#define VNX_MODELNAME		0x22	// get (no set allowed) the device's model name string -- last 6 chars only

#define VNX_DEFAULTS		0x0F	// restore all settings to factory default

// ------------------------ Multi-channel phase shifter Commands ----------------------------------
#define VNX_CHANNEL			0x54	// set the channel, 

//------------------------- Status Report ID Byte -------------------------------------
#define VNX_STATUS			0x52	// Not really a command, but the status byte value for periodic status reports.		

#define VNX_HRSTATUS		0x53	// status report used by HiRes  -- !!Check this, it may be wrong. Not used at the moment!!
#define VNX_HR8STATUS		0x55	// status report used by HiRes
#define VNX_MAXCHANNEL  0x56  // Report the Maxchannels of the device


// ---------------------- LPS system Specific Commands ---------------------
#define VNX_SWVERSION      0x60	    // SW Version
#define VNX_IPMODE         0x61     // IP interface mode Static/DHCP
#define VNX_IPADDR         0x62     // IP Address
#define VNX_IPMASK         0x63     // Subnet Mask
#define VNX_IPGW           0x64     // Gateway

// Device IP length (IPV4)
#define DEVICE_IPV4_LENGTH 16
#define RXBUFF_DATA_LENGTH 24
#define TXBUFF_CMD_LENGTH   8


#define BYTE unsigned char

// Response Data Message Datastructure
typedef struct
{
  BYTE command;
  BYTE length;
  BYTE data[HR_BLOCKSIZE];
} RESPONSE_DATA_T;


// LPS Device Response Data Structure
typedef struct
{
  int ipmode;
  char ipaddress[MAX_NETBUFF];
  char netmask[MAX_NETBUFF];
  char gateway[MAX_NETBUFF];

  //  Global device variables
  int DevType;
  int MinFrequency;
  int MaxFrequency;
  int MinPhase;
  int MaxPhase;			// maximum phase shift in .05 degree units
  int MinPhaseStep;				// replaces DevResolution, smallest phase step in .05 degree units
  int UnitScale;				// the number of .05 degree units in the device's HW representation (1 unit for .05 degrees, 20 units for 1 degree)
  int NumChannels;				// the number of channels the device has including expansion channels
  bool Expandable;				// true for devices that can have expansion channels
  int ProfileMaxLength;			// 25, 50, or 1000 depending on the device
  int ProfileMaxSaved;			// the maximum length of the profile that can be saved in eeprom, 25 for HiRes devices, 50 for other devices
  volatile int Channel;			// the current channel number
  volatile int ChMask;			// our channel in bitmask form (0x01 to 0x80 for single channels)
  volatile int GlobalChannel;	// the current global channel using a zero based index
  volatile int BaseChannels;	// the total number of channels in the base LPS device
  int SerialNumber;
  char ModelName[MAX_MODELNAME];
  char Swversion[MAX_SWVERSION];
  char HaveNumChannels;			// set to a non-zero value if we know the number of channels in the LDA device
  //  Per channel variables
  volatile int DevStatus[CHANNEL_MAX];
  volatile int WorkingFrequency[CHANNEL_MAX];
  volatile int PhaseAngle[CHANNEL_MAX];				// in .05 degree units
  volatile int RampStart[CHANNEL_MAX];
  volatile int RampStop[CHANNEL_MAX];
  volatile int PhaseStep[CHANNEL_MAX];			// ramp step size for the first phase of the ramp
  volatile int PhaseStep2[CHANNEL_MAX];			// ramp step size for second phase of the ramp
  volatile int DwellTime[CHANNEL_MAX];
  volatile int DwellTime2[CHANNEL_MAX];
  volatile int IdleTime[CHANNEL_MAX];
  volatile int HoldTime[CHANNEL_MAX];
  volatile int ProfileCount[CHANNEL_MAX];
  volatile int ProfileIndex[CHANNEL_MAX];
  volatile int ProfileDwellTime[CHANNEL_MAX];
  volatile int ProfileIdleTime[CHANNEL_MAX];
  volatile int Modebits[CHANNEL_MAX];
  volatile int CachedProfileValue[CHANNEL_MAX];			// we only cache the last used profile entry now to avoid allocating a large but rarely used memory buffer
														// we don't use a malloc/free strategy to avoid memory leak issues if an application fails to close a device
														// The upper 16 bits holds the index, the lower 16 bits holds the value. Packing them as a pair into one int avoids possible race conditions
														// due to a non-atomic read of the structure if they were separate variables.

  // Internal variables used to identify and manage the hardware
  char deviceip[DEVICE_IPV4_LENGTH];
  char sndbuff[TXBUFF_CMD_LENGTH];
  char rcvbuff[RXBUFF_DATA_LENGTH];
  int  devicesockfd;
  int  deviceready; // Set the device ready flag 
  int  responseready; // Response ready flag

} LPSPARAMS;

//*****************************************************************************
//
// Mark the end of the C bindings section for C++ compilers.
//
//*****************************************************************************
#ifdef __cplusplus
}
#endif

#endif // __LPSSOCKET_H__
